{
    This file is part of RA1792Control

    RACAL RA1792 Control program

    Copyright (C) 2012-2025 G. Perotti, I1EPJ, i1epj@aricasale.it

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/.

                                    * * *

    Hamlib rigctld-maybe-compatible RA1792 control net server.
    The strictly-networking part is in skanti.pas using lNET.

    Rigctld commands are only partially coded. The implementation uses
    only the basic remote command capabilities of the RA1792.
    ==========================================================================
                                   WANTED
    ==========================================================================
             Someone with better knowledge of how rigctld works
                to help complete/correct the implementation.
    ==========================================================================
}

{$include defines.inc}

unit UParse;

{$mode ObjFPC}

interface

uses
  Classes, SysUtils, UState, StrUtils, lNet;

procedure ParseNetMsg(RXLine: string; MySocket: TLSocket);

// Messages declared here for i18n
resourcestring
  REMCMDNOTREC='Remote command not recognized: %s';
  REMCMDNOTIMP='Remote command not implemented: %s';
  REMCMDNOTAVA='Remote command not available: %s';
  EXTPROTNOTSUPP='Extended response protocol is not supported';
  SENTHAMLIBCMDS='Sent: %s to port %d';
  RECVHAMLIBCMDS='Recv: %s %s %s from port %d';

implementation

uses ura1792;

type

  // The hamlib rigtcld commands
  TCmdCodes =
    (SET_FREQ, GET_FREQ, SET_MODE, GET_MODE, SET_VFO, GET_VFO,
     SET_RIT, GET_RIT, SET_XIT, GET_XIT, SET_PTT, GET_PTT,
     SET_SPLIT_VFO, GET_SPLIT_VFO, SET_SPLIT_FREQ, GET_SPLIT_FREQ,
     SET_SPLIT_MODE, GET_SPLIT_MODE, SET_ANT, GET_ANT, SEND_MORSE,
     GET_DCD, SET_RPTR_SHIFT, GET_RPTR_SHIFT, SET_RPTR_OFFS,GET_RPTR_OFFS,
     SET_CTCSS_TONE, GET_CTCSS_TONE, SET_DCS_CODE, GET_DCS_CODE,
     SET_CTCSS_SQL, GET_CTCSS_SQL, SET_DCS_SQL, GET_DCS_SQL, SET_TS,
     GET_TS, SET_FUNC, GET_FUNC, SET_LEVEL, GET_LEVEL, SET_PARM, GET_PARM,
     SET_BANK, SET_MEM, GET_MEM, VFO_OP, SCAN, SET_CHANNEL, GET_CHANNEL,
     SET_TRN, GET_TRN, RESET, SET_POWERSTAT, GET_POWERSTAT, SEND_DTMF,
     RECV_DTMF, GET_INFO, GET_RIG_INFO, GET_VFO_INFO, DUMP_STATE,DUMP_CAPS,
     POWER2MW, MW2POWER, SET_CLOCK, GET_CLOCK, CHK_VFO, SET_VFO_OPT, QUIT,
     SET_LOCK_MODE,GET_LOCK_MODE,LASTCMD
  );

  // The hamlib rigtcld VFOs
  TVFONames =
    (NONE, VFOA, VFOB, VFOC, currVFO, VFO, MEM, Main, Sub, TX, RX);

  // The hamlib rigtcld levels
  TLevels = (PREAMP,ATT,LVOX,AF,LRF,LSQL,LIF,LAPF,LNR,PBT_IN,PBT_OUT,CWPITCH,RFPOWER,
     RFPOWER_METER,RFPOWER_METER_WATTS,MICGAIN,KEYSPD,NOTCHF,LCOMP,AGC,BKINDL,BAL,
     METER,VOXGAIN,ANTIVOX,SLOPE_LOW,SLOPE_HIGH,RAWSTR,SWR,ALC,STRENGTH,
     LASTLEVEL);

  // The hamlib rigtcld functions
  TFunctions =
    (FAGC,NB,COMP,FVOX,TONE,TSQL,SBKIN,FBKIN,ANF,NR,AIP,APF,MON,MN,FRF,ARO,LOCK,
     MUTE,VSC,REV,FSQL,ABM,BC,MBC,RIT,AFC,SATMODE,SCOPE,RESUME,TBURST,TUNER,XIT,
     LASTFUNCTION
    );

  // Hamlib error codes
  // Taken from hamlib sources (rig.h)
  THamlibErrors = (
    RIG_OK,          // 0 No error, operation completed successfully */
    RIG_EINVAL,      // 1 invalid parameter */
    RIG_ECONF,       // 2 invalid configuration (serial,..) */
    RIG_ENOMEM,      // 3 memory shortage */
    RIG_ENIMPL,      // 4 function not implemented, but will be */
    RIG_ETIMEOUT,    //  5 communication timed out */
    RIG_EIO,         //  6 IO error, including open failed */
    RIG_EINTERNAL,   //  7 Internal Hamlib error, huh! */
    RIG_EPROTO,      //  8 Protocol error */
    RIG_ERJCTED,     //  9 Command rejected by the rig */
    RIG_ETRUNC,      //  10 Command performed, but arg truncated */
    RIG_ENAVAIL,     //  11 Function not available */
    RIG_ENTARGET,    //  12 VFO not targetable */
    RIG_BUSERROR,    //  13 Error talking on the bus */
    RIG_BUSBUSY,     //  14 Collision on the bus */
    RIG_EARG,        //  15 NULL RIG handle or any invalid pointer parameter in
    RIG_EVFO,        //  16 Invalid VFO */
    RIG_EDOM,        //  17 Argument out of domain of func */
    RIG_EDEPRECATED, //  18 Function deprecated */
    NOTREQUIRED      //  No error code required
  );

  // The command parser structure
  THamLibCmd = record
    LongName: string;
    ShortName: string;
    CmdCode: TCmdCodes;
  end;

  // The level parser structure
  THamlibLevel = record
    LevelName: string;
    LevelCode: TLevels;
  end;

  // The function parser structure
  THamlibFunction = record
    FunctionName: string;
    FunctionCode: TFunctions;
  end;

  // The commands array
  THamlibCmds = array[1..ord(LASTCMD)] of THamLibCmd;

  // The levels array
  THamlibLevels = array[1..ord(LASTLEVEL)] of THamlibLevel;

  // The functions array
  THamlibFunctions = array[1..ord(LASTFUNCTION)] of THamlibFunction;

  // Various constant values
  const
    RPRT: string = 'RPRT ';
    RPRT_OK: string = 'RPRT 0';
    VFOMode: integer = 0;

  HamLibCmds: THamlibCmds = (
    (LongName:'set_freq';ShortName:'F';CmdCode:SET_FREQ),
    (LongName:'get_freq';ShortName:'f';CmdCode:GET_FREQ),
    (LongName: 'set_lock_mode'; ShortName:''; CmdCode: SET_LOCK_MODE),
    (LongName: 'get_lock_mode'; ShortName:''; CmdCode: GET_LOCK_MODE),
    (LongName:'set_mode';ShortName:'M';CmdCode:SET_MODE),
    (LongName:'get_mode';ShortName:'m';CmdCode:GET_MODE),
    (LongName:'set_vfo';ShortName:'V';CmdCode:SET_VFO),
    (LongName:'get_vfo';ShortName:'v';CmdCode:GET_VFO),
    (LongName:'set_rit';ShortName:'J';CmdCode:SET_RIT),
    (LongName:'get_rit';ShortName:'j';CmdCode:GET_RIT),
    (LongName:'set_xit';ShortName:'Z';CmdCode:SET_XIT),
    (LongName:'het_xit';ShortName:'z';CmdCode:GET_XIT),
    (LongName:'set_ptt';ShortName:'T';CmdCode:SET_PTT),
    (LongName:'get_ptt';ShortName:'t';CmdCode:GET_PTT),
    (LongName:'set_split_vfo';ShortName:'S';CmdCode:SET_SPLIT_VFO),
    (LongName:'get_split_vfo';ShortName:'s';CmdCode:GET_SPLIT_VFO),
    (LongName:'set_split_freq';ShortName:'I';CmdCode:SET_SPLIT_FREQ),
    (LongName:'get_split_freq';ShortName:'i';CmdCode:GET_SPLIT_FREQ),
    (LongName:'set_split_mode';ShortName:'X';CmdCode:SET_SPLIT_MODE),
    (LongName:'get_split_mode';ShortName:'x';CmdCode:GET_SPLIT_MODE),
    (LongName:'set_ant';ShortName:'Y';CmdCode:SET_ANT),
    (LongName:'get_ant';ShortName:'y';CmdCode:GET_ANT),
    (LongName:'send_morse';ShortName:'b';CmdCode:SEND_MORSE),
    (LongName:'get_dcd';ShortName:#$8B;CmdCode:GET_DCD),
    (LongName:'set_rptr_shift';ShortName:'R';CmdCode:SET_RPTR_SHIFT),
    (LongName:'get_rptr_shift';ShortName:'r';CmdCode:GET_RPTR_SHIFT),
    (LongName:'set_rptr_offs';ShortName:'O';CmdCode:SET_RPTR_OFFS),
    (LongName:'get_rptr_offs';ShortName:'o';CmdCode:GET_RPTR_OFFS),
    (LongName:'set_ctcss_tone';ShortName:'C';CmdCode:SET_CTCSS_TONE),
    (LongName:'get_ctcss_tone';ShortName:'c';CmdCode:GET_CTCSS_TONE),
    (LongName:'set_dcs_code';ShortName:'D';CmdCode:SET_DCS_CODE),
    (LongName:'get_dcs_code';ShortName:'d';CmdCode:GET_DCS_CODE),
    (LongName:'set_ctcss_sql';ShortName:#$90;CmdCode:SET_CTCSS_SQL),
    (LongName:'get_ctcss_sql';ShortName:#$91;CmdCode:GET_CTCSS_SQL),
    (LongName:'set_dcs_sql';ShortName:#$92;CmdCode:SET_DCS_SQL),
    (LongName:'get_dcs_sql';ShortName:#$93;CmdCode:GET_DCS_SQL),
    (LongName:'set_ts';ShortName:'N';CmdCode:SET_TS),
    (LongName:'get_ts';ShortName:'n';CmdCode:GET_TS),
    (LongName:'set_func';ShortName:'U';CmdCode:SET_FUNC),
    (LongName:'get_func';ShortName:'u';CmdCode:GET_FUNC),
    (LongName:'set_level';ShortName:'L';CmdCode:SET_LEVEL),
    (LongName:'get_level';ShortName:'l';CmdCode:GET_LEVEL),
    (LongName:'set_parm';ShortName:'P';CmdCode:SET_PARM),
    (LongName:'get_parm';ShortName:'p';CmdCode:GET_PARM),
    (LongName:'set_bank';ShortName:'B';CmdCode:SET_BANK),
    (LongName:'set_mem';ShortName:'E';CmdCode:SET_MEM),
    (LongName:'get_mem';ShortName:'e';CmdCode:GET_MEM),
    (LongName:'vfo_op';ShortName:'G';CmdCode:VFO_OP),
    (LongName:'scan';ShortName:'g';CmdCode:SCAN),
    (LongName:'set_channel';ShortName:'H';CmdCode:SET_CHANNEL),
    (LongName:'get_channel';ShortName:'h';CmdCode:GET_CHANNEL),
    (LongName:'set_trn';ShortName:'A';CmdCode:SET_TRN),
    (LongName:'get_trn';ShortName:'a';CmdCode:GET_TRN),
    (LongName:'reset';ShortName:'*';CmdCode:RESET),
    (LongName:'set_powerstat';ShortName:#$87;CmdCode:SET_POWERSTAT),
    (LongName:'get_powerstat';ShortName:#$88;CmdCode:GET_POWERSTAT),
    (LongName:'send_dtmf';ShortName:#$89;CmdCode:SEND_DTMF),
    (LongName:'recv_dtmf';ShortName:#$8A;CmdCode:RECV_DTMF),
    (LongName:'get_info';ShortName:'_';CmdCode:GET_INFO),
    (LongName:'get_rig_info';ShortName:#$F5;CmdCode:GET_RIG_INFO),
    (LongName:'get_vfo_info';ShortName:#$F3;CmdCode:GET_VFO_INFO),
    (LongName:'dump_state';ShortName:'';CmdCode:DUMP_STATE),
    (LongName:'dump_caps';ShortName:'1';CmdCode:DUMP_CAPS),
    (LongName:'power2mW';ShortName:'2';CmdCode:POWER2MW),
    (LongName:'mW2power';ShortName:'4';CmdCode:MW2POWER),
    (LongName:'set_clock';ShortName:'';CmdCode:SET_CLOCK),
    (LongName:'get_clock';ShortName:'';CmdCode:GET_CLOCK),
    (LongName:'chk_vfo';ShortName:'';CmdCode:CHK_VFO),
    (LongName:'set_vfo_opt';ShortName:'';CmdCode:SET_VFO_OPT),
    (LongName:'quit';ShortName:'q';CmdCode:QUIT)
  );

  HamlibLevels: THamlibLevels = (
    (LevelName: 'PREAMP'; LevelCode: PREAMP),
    (LevelName: 'ATT'; LevelCode: ATT),
    (LevelName: 'VOX'; LevelCode: LVOX),
    (LevelName: 'AF'; LevelCode: AF),
    (LevelName: 'RF'; LevelCode: LRF),
    (LevelName: 'SQL'; LevelCode: LSQL),
    (LevelName: 'IF'; LevelCode: LIF),
    (LevelName: 'APF'; LevelCode: LAPF),
    (LevelName: 'NR'; LevelCode: LNR),
    (LevelName: 'PBT_IN'; LevelCode: PBT_IN),
    (LevelName: 'PBT_OUT'; LevelCode: PBT_OUT),
    (LevelName: 'CWPITCH'; LevelCode: CWPITCH),
    (LevelName: 'RFPOWER'; LevelCode: RFPOWER),
    (LevelName: 'RFPOWER_METER'; LevelCode: RFPOWER_METER),
    (LevelName: 'RFPOWER_METER_WATTS'; LevelCode: RFPOWER_METER_WATTS),
    (LevelName: 'MICGAIN'; LevelCode: MICGAIN),
    (LevelName: 'KEYSPD'; LevelCode: KEYSPD),
    (LevelName: 'NOTCHF'; LevelCode: NOTCHF),
    (LevelName: 'COMP'; LevelCode: NOTCHF),
    (LevelName: 'AGC'; LevelCode: AGC),
    (LevelName: 'BKINDL'; LevelCode: BKINDL),
    (LevelName: 'BAL'; LevelCode: BAL),
    (LevelName: 'METER'; LevelCode: METER),
    (LevelName: 'VOXGAIN'; LevelCode: VOXGAIN),
    (LevelName: 'ANTIVOX'; LevelCode: ANTIVOX),
    (LevelName: 'SLOPE_LOW'; LevelCode: SLOPE_LOW),
    (LevelName: 'SLOPE_HIGH'; LevelCode: SLOPE_HIGH),
    (LevelName: 'RAWSTR'; LevelCode: RAWSTR),
    (LevelName: 'SWR'; LevelCode: SWR),
    (LevelName: 'ALC'; LevelCode: ALC),
    (LevelName: 'STRENGTH'; LevelCode: STRENGTH)
  );

  HamlibFunctions: THamlibFunctions = (
   (FunctionName: 'FAGC'; FunctionCode: FAGC),
   (FunctionName: 'NB'; FunctionCode: NB),
   (FunctionName: 'COMP'; FunctionCode: COMP),
   (FunctionName: 'VOX'; FunctionCode: FVOX),
   (FunctionName: 'TONE'; FunctionCode: TONE),
   (FunctionName: 'TSQL'; FunctionCode: TSQL),
   (FunctionName: 'SBKIN'; FunctionCode: SBKIN),
   (FunctionName: 'FBKIN'; FunctionCode: FBKIN),
   (FunctionName: 'ANF'; FunctionCode: ANF),
   (FunctionName: 'NR'; FunctionCode: NR),
   (FunctionName: 'AIP'; FunctionCode: AIP),
   (FunctionName: 'APF'; FunctionCode: APF),
   (FunctionName: 'MON'; FunctionCode: MON),
   (FunctionName: 'MN'; FunctionCode: MN),
   (FunctionName: 'RF'; FunctionCode: FRF),
   (FunctionName: 'ARO'; FunctionCode: ARO),
   (FunctionName: 'LOCK'; FunctionCode: LOCK),
   (FunctionName: 'MUTE'; FunctionCode: MUTE),
   (FunctionName: 'VSC'; FunctionCode: VSC),
   (FunctionName: 'REV'; FunctionCode: REV),
   (FunctionName: 'SQL'; FunctionCode: FSQL),
   (FunctionName: 'ABM'; FunctionCode: ABM),
   (FunctionName: 'BC'; FunctionCode: BC),
   (FunctionName: 'MBC'; FunctionCode: MBC),
   (FunctionName: 'RIT'; FunctionCode: RIT),
   (FunctionName: 'AFC'; FunctionCode: AFC),
   (FunctionName: 'SATMODE'; FunctionCode: SATMODE),
   (FunctionName: 'SCOPE'; FunctionCode: SCOPE),
   (FunctionName: 'RESUME'; FunctionCode: RESUME),
   (FunctionName: 'TBURST'; FunctionCode: TBURST),
   (FunctionName: 'TUNER'; FunctionCode: TUNER),
   (FunctionName: 'XIT'; FunctionCode: XIT)
  );

// Global variables
var
  SplitVFOActive: boolean;
  curVFO: TVFONames = VFOA;
  tmpVFO: TVFONames = NONE;
  dbS9: integer;
  HamLibResult: THamlibErrors;
  Lock_Mode: integer = 0;


// Auxiliary procedures & functions

// Send a string via TCP socket
procedure SendNetMsg(Msg:string; MySocket: TLSocket);

var s: string;

begin
  Msg := Msg+stLF;
  MySocket.SendMessage(Msg);
  if RA1792.MSHOWHAMLIB.Checked then begin
    s := Format(SENTHAMLIBCMDS,[RA1792.ShowControlChars(Msg), MySocket.LocalPort]);
    RA1792.MsgToShow := s;
  end;
end;

// Parse and execute hamlib commands
procedure ParseNetMsg(RXLine: string; MySocket: TLSocket);

var LineNum, ParamNum, NewFreq: integer;
    sCmd, sParam1, sParam2, sTmp: string;
    {$IFDEF THIRDPARAMETER}
    // not required so far
    sParam3: string;
    {$ENDIF}
    Cmd: TCmdCodes = LASTCMD;
    Level: TLevels = LASTLEVEL;
    Functn: TFunctions = LASTFUNCTION;

begin
  {$IFDEF DEBUG_PARSE}
  RA1792.MsgToShow := LineEnding+'ParseNetMsg entered';
  RA1792.MsgToShow := 'Line received: '+RA1792.ShowControlChars(RXLine);
  {$ENDIF}

  if RXLine='' then exit;

  Cmd := LASTCMD;

  // Remove trailing/ending white spaces and control chars
  RXLine := Trim(RXLine);

  // Separate command and parameters, if any, and remove remaining spaces
  // and control chars. Handle either LF and space delimiters.
  sCmd := Trim(ExtractDelimited(1,RXLine,[' ',#$0a]));
  sParam1 := Trim(ExtractDelimited(2,RXLine,[' ',#$0a]));
  sParam2 := Trim(ExtractDelimited(3,RXLine,[' ',#$0a]));
  {$IFDEF THIRDPARAMETER}
  // not required so far
  sParam3 := Trim(ExtractDelimited(4,RXLine,[' ',#$0a]));
  {$ENDIF}
  {$IFDEF DEBUG_PARSE}
  RA1792.MsgToShow := 'Cmd: '+sCmd;
  RA1792.MsgToShow := 'Param1: '+sParam1;
  RA1792.MsgToShow := 'Param2: '+sParam2;
  {$IFDEF THIRDPARAMETER}
  // not required so far
  RA1792.MsgToShow := 'Param3: '+sParam3;
  {$ENDIF}
  {$ENDIF}

  // Blank command, ignore it
  if sCmd = '' then exit;

   // Check for extended response protocol (NOT SUPPORTED)
  if sCmd[1] in ['+',';','|',','] then begin
    RA1792.MsgToShow := EXTPROTNOTSUPP;
    SendNetMsg(RPRT+IntToStr(-Ord(RIG_ENIMPL)),MySocket);
    exit;
  end;

  // Parse hamlib long commands
  if sCmd[1]='\' then begin
    // It is a long command
    Delete(sCmd,1,1);
    for ParamNum := 1 to Ord(LASTCMD) do begin
      if sCmd=HamLibCmds[ParamNum].LongName then begin
        Cmd := HamLibCmds[ParamNum].CmdCode;
        break;
      end;
    end;
  end else begin
    // parse hamlib short commands
    for ParamNum := 1 to ord(LASTCMD) do begin
      if sCmd=HamLibCmds[ParamNum].ShortName then begin
         Cmd := HamLibCmds[ParamNum].CmdCode;
         break;
      end;
    end;
  end;

  // If "Show hamlib commands" option checked, show received command
  if RA1792.MSHOWHAMLIB.Checked then begin
    sTmp := Format(RECVHAMLIBCMDS, [HamLibCmds[ParamNum].LongName, sParam1, sParam2,
                   MySocket.LocalPort]);
    RA1792.MsgToShow := sTmp;
  end;

  // Now we may have a command in Cmd with parameters in sParam1, sParam2 and sParam3

  // Execute command
  // FIXME: At present only VFOMode 0 is handled.
  // Any program out there *wants* VFOMODE 1?
  HamLibResult := NOTREQUIRED;
  case Cmd of
    SET_FREQ: begin
      // Frequency can have decimal figures (not used here) so remove them if present
      // The separator can be '.' or ',', so handle both
      sParam1 := ExtractDelimited(1,sParam1,['.',',']);
      if VFOMode=0 then begin
        if not In_Command then begin
          // The only handled so far
          if TryStrToInt(sParam1, NewFreq) then begin
            // Check if frequency is in allowable range
            // if outside, set it to lowest/highest allowed value
            if NewFreq < MINRXF then NewFreq := MINRXF;
            if NewFreq > MAXRXF then NewFreq := MAXRXF;

            In_Command := TRUE;
            RXState.Freq_Rx := NewFreq;
            RA1792.SetRXFreq;
            PanelUpdateRequired := TRUE;
            In_Command := FALSE;
            HamLibResult := RIG_OK;
          end else HamLibResult := RIG_EINVAL;
        end else HamLibResult := RIG_BUSBUSY;
      end;
    end;
    GET_FREQ: begin
      // TBD correctly for VFOmode 1 if required
      if VFOMode=0 then
        SendNetMsg(IntToStr(RXState.Freq_rx),MySocket);
    end;
    SET_MODE: begin
      if not In_Command then begin
        // To be completed if needed and possible for all modes supported
        if sParam1='USB' then RA1792.BUSBClick(nil)
        else if sParam1='LSB' then RA1792.BLSBClick(nil)
        else if sParam1='CW' then RA1792.BCWClick(nil)
        else if sParam1='AM' then RA1792.BAMClick(nil)
        else if sParam1='FM' then RA1792.BFMClick(nil);

        // For CW, AM and FM modes only bandwidth is selectable
        if (RXState.Mode_RX = M_CW) or (RXState.Mode_RX = M_AM) or
          (RXState.Mode_RX = M_FM) then begin
          if sParam2 = '16000' then RA1792.B5Click(nil)
          else if sParam2 = '6000' then RA1792.B4Click(nil)
          else if sParam2 = '3200' then RA1792.B3Click(nil)
          else if sParam2 = '1000' then RA1792.B2Click(nil)
          else if sParam2 = '300' then RA1792.B1Click(nil)
        end;
        HamLibResult := RIG_OK;
      end else HamLibResult := RIG_BUSBUSY;
    end;
    GET_MODE: begin
      sTmp := '';
      case RXState.Mode_RX of
        M_USB: sTmp:='USB';
        M_LSB: sTmp:='LSB';
        M_FM: sTmp:='FM';
        M_CW: sTmp:='CW';
        M_AM: sTmp:='AM';
        // FIXME: what to return for this mode?! ISB is not a rigctld valid
        // mode token, so return DSB even if it is not correct to avoid
        // misbeahving of other programs
        M_ISB:sTmp:='DSB';
      end;
      // rigctld returns <mode><LF><bandwidth>, so the same is done here.
      sTmp := sTmp+stLF;

      // Values are for my own RA1792
      case RXState.Filter of
        FILTER_300: sTmp := sTmp+'300';
        FILTER_1000: sTmp := sTmp+'1000';
        FILTER_3200: sTmp := sTmp+'3200';
        FILTER_6000: sTmp := sTmp+'6000';
        FILTER_16000: sTmp := sTmp+'16000';
      end;
      SendNetMsg(sTmp,MySocket);
    end;
    SET_VFO: begin
      // Only one VFO available. Report always success.
      HamLibResult := RIG_OK;
    end;
    GET_VFO: begin
      // Only one VFO available, report VFOA.
      SendNetMsg('VFOA',MySocket);
    end;
    SET_RIT: begin
      // Not available
      RA1792.MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamLibResult := RIG_ENAVAIL;
    end;
    GET_RIT: begin
      // Not available
      RA1792.MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamLibResult := RIG_ENAVAIL;
    end;
    SET_XIT: begin
      // Not available
      RA1792.MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamLibResult := RIG_ENAVAIL;
    end;
    GET_XIT: begin
      // Not available
      RA1792.MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamLibResult := RIG_ENAVAIL;
    end;
    SET_PTT: begin
      // Does not apply to a receiver but other programs don't know that,
      // so ignore it and send a OK reply
      HamLibResult := RIG_OK;
    end;
    GET_PTT: begin
      // Does not apply to a receiver but other programs don't know that,
      // so send a '0' reply (RX)
      SendNetMsg('0',MySocket);
    end;
    SET_SPLIT_VFO: begin
      // Does not apply
      RA1792.MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamLibResult := RIG_ENAVAIL;
    end;
    GET_SPLIT_VFO: begin
      // No split VFO and VFOA active
      SendNetMSG('0'+stLF+'VFOA',MySocket);
    end;
    SET_SPLIT_FREQ: begin
      // Does not apply
      RA1792.MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamLibResult := RIG_ENAVAIL;
    end;
    GET_SPLIT_FREQ: begin
      // Does not apply
      RA1792.MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamLibResult := RIG_ENAVAIL;
    end;
    SET_SPLIT_MODE: begin
      // Does not apply
      RA1792.MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamLibResult := RIG_ENAVAIL;
    end;
    GET_SPLIT_MODE: begin
     // Does not apply
      RA1792.MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamLibResult := RIG_ENAVAIL;
    end;
    SET_ANT: begin
      // Not available
      RA1792.MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamLibResult := RIG_ENAVAIL;
    end;
    GET_ANT: begin
      // Not available
      RA1792.MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamLibResult := RIG_ENAVAIL;
    end;
    SEND_MORSE: begin
      // Not available
      RA1792.MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamLibResult := RIG_ENAVAIL;
    end;
    GET_DCD: begin
      // Not available
      RA1792.MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamLibResult := RIG_ENAVAIL;
    end;
    SET_RPTR_SHIFT: begin
      // Not available
      RA1792.MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamLibResult := RIG_ENAVAIL;
    end;
    GET_RPTR_SHIFT: begin
      // Not available
      RA1792.MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamLibResult := RIG_ENAVAIL;
    end;
    SET_RPTR_OFFS: begin
      // Not available
      RA1792.MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamLibResult := RIG_ENAVAIL;
    end;
    GET_RPTR_OFFS: begin
      // Not available
      RA1792.MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamLibResult := RIG_ENAVAIL;
    end;
    SET_CTCSS_TONE: begin
      // Not available
      RA1792.MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamLibResult := RIG_ENAVAIL;
    end;
    GET_CTCSS_TONE: begin
      // Not available
      RA1792.MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamLibResult := RIG_ENAVAIL;
    end;
    SET_DCS_CODE: begin
      // Not available
      RA1792.MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamLibResult := RIG_ENAVAIL;
    end;
    GET_DCS_CODE: begin
      // Not available
      RA1792.MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamLibResult := RIG_ENAVAIL;
    end;
    SET_CTCSS_SQL: begin
      // Not available
      RA1792.MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamLibResult := RIG_ENAVAIL;
    end;
    GET_CTCSS_SQL: begin
      // Not available
      RA1792.MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamLibResult := RIG_ENAVAIL;
    end;
    SET_DCS_SQL: begin
      // Not available
      RA1792.MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamLibResult := RIG_ENAVAIL;
    end;
    GET_DCS_SQL: begin
      // Not available
      RA1792.MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamLibResult := RIG_ENAVAIL;
    end;
    SET_TS: begin
      // RA1792 has no tune steps, it is continuously tunable in 10Hz
      // (most models) or 1Hz (frequency measuring version) increments.
      RA1792.MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamLibResult := RIG_ENAVAIL;
    end;
    GET_TS: begin
      // RA1792 has no tune steps, it is continuously tunable in 10Hz
      // (most models) or 1Hz (frequency measuring version) increments.
      // Returns the *program" tune step.
      SendNetMsg(IntToStr(RXState.Freq_step), MySocket);
    end;
    SET_FUNC: begin
      // To be completed with all applicable functions
      for ParamNum := 1 to ord(LASTFUNCTION) do begin
        if sParam1=HamlibFunctions[ParamNum].FunctionName then begin
          Functn := HamlibFunctions[ParamNum].FunctionCode;
          break;
        end;
      end;
      // No functions
    end;
    GET_FUNC: begin
       // To be completed with all applicable functions
      for ParamNum := 1 to ord(LASTFUNCTION) do begin
        if sParam1=HamlibFunctions[ParamNum].FunctionName then begin
          Functn := HamlibFunctions[ParamNum].FunctionCode;
          break;
        end;
      end;
      // No functions
    end;
    SET_LEVEL: begin
      // To be completed with all applicable levels
      for ParamNum := 1 to ord(LASTLEVEL) do begin
        if sParam1=HamlibLevels[ParamNum].LevelName then begin
          Level := HamlibLevels[ParamNum].LevelCode;
          break;
        end;
      end;

      if not (In_Command or In_Status) then begin
        case Level of
          AGC: begin
            HamLibResult := RIG_OK;
            case StrToInt(sParam2) of
              0: RA1792.B6Click(nil);
              1,2: RA1792.B7Click(nil);
              3: RA1792.B9Click(nil);
              5: RA1792.B8Click(nil);
              otherwise HamLibResult := RIG_EINVAL;
            end;
          end;
          otherwise HamLibResult := RIG_EINVAL;
        end;
      end else HamLibResult := RIG_BUSBUSY;
    end;
    GET_LEVEL: begin
      // To be completed with all applicable levels
      for ParamNum := 1 to ord(LASTLEVEL) do begin
        if sParam1=HamlibLevels[ParamNum].LevelName then begin
          Level := HamlibLevels[ParamNum].LevelCode;
          break;
        end;
      end;

      case Level of
        AGC: begin
          case RXstate.AGC of
            AGC_MAN: SendNetMsg('0',MySocket);
            AGC_SHORT, AGC_MAN_SHORT: SendNetMsg('2',MySocket);
            AGC_MEDIUM, AGC_MAN_MEDIUM: SendNetMsg('5',MySocket);
            AGC_LONG, AGC_MAN_LONG: SendNetMsg('3',MySocket);
          end;
        end;
        STRENGTH: begin
          // SignalLevel per manual is 0-150, scale it to 0-120dBuV
          // The hamlib doc is unclear on which units should be used, seems dBS9.
          // 0 dBuV is 1 uV, S3 is about 1.5 uV, S2 is 0,75 uV.
          // With a returned value of -40 grig S-Meter shows a little more
          // than S2, which seems reasonable.
          // NB NOT TESTED, my RA1792 lacks the TR command.
          dBS9 := (120*SignalLevel div 150)-40;
          SendNetMsg(IntToStr(dBS9),MySocket);
        end;
      end;
    end;
    SET_PARM: begin
      // TBD if some program *needs* it
      RA1792.MsgToShow := Format(REMCMDNOTIMP,[sCmd]);
      HamLibResult := RIG_ENIMPL;
    end;
    GET_PARM: begin
      // TBD if some program *needs* it
      RA1792.MsgToShow := Format(REMCMDNOTIMP,[sCmd]);
      HamLibResult := RIG_ENIMPL;
    end;
    SET_BANK: begin
      // Not available
      RA1792.MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamLibResult := RIG_ENAVAIL;
    end;
    SET_MEM: begin
      // TBD if some program *needs* it
      RA1792.MsgToShow := Format(REMCMDNOTIMP,[sCmd]);
      HamLibResult := RIG_ENAVAIL;
    end;
    GET_MEM: begin
      // TBD if some program *needs* it
      RA1792.MsgToShow := Format(REMCMDNOTIMP,[sCmd]);
      HamLibResult := RIG_ENAVAIL;
    end;
    VFO_OP: begin
      // TBD for ops supported, if any, and if *needed* by some program
      RA1792.MsgToShow := Format(REMCMDNOTIMP,[sCmd]);
      HamLibResult := RIG_ENIMPL;
    end;
    SCAN: begin
      // TBD if some program *needs* it
      RA1792.MsgToShow := Format(REMCMDNOTIMP,[sCmd]);
      HamLibResult := RIG_ENIMPL;
    end;
    SET_CHANNEL: begin
      // TBD if some program *needs* it
      RA1792.MsgToShow := Format(REMCMDNOTIMP,[sCmd]);
      HamLibResult := RIG_ENIMPL;
    end;
    GET_CHANNEL: begin
      // TBD if some program *needs* it
      RA1792.MsgToShow := Format(REMCMDNOTIMP,[sCmd]);
      HamLibResult := RIG_ENIMPL;
    end;
    SET_TRN: begin
      // Not available (?)
      RA1792.MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamLibResult := RIG_ENAVAIL;
    end;
    GET_TRN: begin
      // Not available (?)
      RA1792.MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamLibResult := RIG_ENAVAIL;
    end;
    RESET: begin
      // Not available
      RA1792.MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamLibResult := RIG_ENAVAIL;
    end;
    SET_POWERSTAT: begin
      // Not available
      RA1792.MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamLibResult := RIG_ENAVAIL;
    end;
    GET_POWERSTAT: begin
      // wsjtx 2.7.0-rc6 use this command. Reply always 1 (=power on)
      SendNetMsg('1',MySocket);
    end;
    SEND_DTMF: begin
      // Not available
      RA1792.MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamLibResult := RIG_ENAVAIL;
    end;
    RECV_DTMF: begin
      // Not available
      RA1792.MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamLibResult := RIG_ENAVAIL;
    end;
    GET_INFO: begin
      // TBD if some program *needs* it
      RA1792.MsgToShow := Format(REMCMDNOTIMP,[sCmd]);
      HamLibResult := RIG_ENAVAIL;
    end;
    GET_RIG_INFO: begin
      // TBD if some program *needs* it
      RA1792.MsgToShow := Format(REMCMDNOTIMP,[sCmd]);
      HamLibResult := RIG_ENIMPL;
    end;
    GET_VFO_INFO: begin
      // TBD if some program *needs* it
      RA1792.MsgToShow := Format(REMCMDNOTIMP,[sCmd]);
      HamLibResult := RIG_ENIMPL;
    end;
    DUMP_STATE: begin
      // TBD in the correct way - See UState.pas
      {$IFDEF SENDDONE}
            LineNum := -1;
            repeat
              LineNum := Linenum + 1;
              SendNetMsg(RA1792_State[LineNum],MySocket);
            until RA1792_State[LineNum]='done';
      {$ELSE}
      LineNum := 0;
      repeat
        SendNetMsg(TRP8255_State[LineNum],MySocket);
        LineNum := Linenum + 1;
      until RA1792_State[LineNum]='done';
      {$ENDIF}
    end;
    DUMP_CAPS: begin
      // TBD if required
      RA1792.MsgToShow := Format(REMCMDNOTIMP,[sCmd]);
      HamLibResult := RIG_ENIMPL;
    end;
    POWER2MW: begin
      // Does not apply
      RA1792.MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamLibResult := RIG_ENAVAIL;
    end;
    MW2POWER: begin
      // Does not apply
      RA1792.MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamLibResult := RIG_ENAVAIL;
    end;
    SET_CLOCK: begin
      // No RTC available
      RA1792.MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamLibResult := RIG_ENAVAIL;
    end;
    GET_CLOCK: begin
      // No RTC available
      RA1792.MsgToShow := Format(REMCMDNOTAVA,[sCmd]);
      HamLibResult := RIG_ENAVAIL;
    end;
    CHK_VFO: begin
      // Required by WSJTX and other programs
      // Only Mode 0 at present is handled, so VFOMode is always 0
      SendNetMsg(IntToStr(VFOMode),MySocket);
    end;
    SET_VFO_OPT: begin
      // TDB if any and if it is *needed* by some program
      RA1792.MsgToShow := Format(REMCMDNOTIMP,[sCmd]);
      HamLibResult := RIG_ENIMPL;
    end;
    QUIT: begin
      SendNetMsg(RPRT_OK,MySocket);
      MySocket.Disconnect(FALSE);
      PanelUpdateRequired := TRUE;
    end;
    SET_LOCK_MODE: begin
      if sParam1[1] in ['0'..'9'] then begin
        Lock_Mode := StrToInt(sParam1[1]);
        HamlibResult := RIG_OK;
      end else HamLibResult := RIG_EINVAL;
    end;
    GET_LOCK_MODE: begin
      SendNetMsg(IntToStr(Lock_Mode), MySocket);
      {$IFDEF SENDRPRT0}
      HamlibResult := RIG_OK;     // Required. Don't know why.
      {$ENDIF}
    end;
    otherwise begin
      // command not recognized
      HamLibResult := RIG_EINVAL;
      RA1792.MsgToShow := Format(REMCMDNOTREC,[sCmd]);
    end;
  end;
  if HamlibResult <> NOTREQUIRED then
    SendNetMsg(RPRT+IntToStr(-Ord(HamlibResult)),MySocket);
end;

end.

