{
    This file is part of EKD500Control

    RFT EKD500 Control program

    Copyright (C) 2013-2022 G. Perotti, I1EPJ, i1epj@aricasale.it

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/.
}

//
// Band scan graph code
//

unit uscan;

{$mode objfpc}

{$INCLUDE defines.inc}
interface

uses
  Classes, SysUtils, LResources, Forms, Controls, Graphics, Dialogs, StdCtrls,
  JLabeledIntegerEdit, JButton, TAGraph, TASeries, lclproc, LazUtilities, StrUtils;

type

  { TFSCAN }

  TFSCAN = class(TForm)
    BLOADCSV: TJButton;
    BSAVESCAN: TJButton;
    BCLOSEW: TJButton;
    BSTARTScaN: TJButton;
    BSTOPSCAN: TJButton;
    BCLEARSCAN: TJButton;
    CBAGC: TComboBox;
    DSAVECSV: TSaveDialog;
    LBAGC: TLabel;
    DOPENCSV: TOpenDialog;
    ScanPlot: TChart;
    CBBANDW: TComboBox;
    CBMODE: TComboBox;
    LBBANDW: TLabel;
    LBMODE: TLabel;
    SignalLevel: TLineSeries;
    IEFSTART: TJLabeledIntegerEdit;
    IEFSTOP: TJLabeledIntegerEdit;
    IEFSTEP: TJLabeledIntegerEdit;
    procedure BLOADCSVClick(Sender: TObject);
    procedure BSAVESCANClick(Sender: TObject);
    procedure BCLEARSCANClick(Sender: TObject);
    procedure BCLOSEWClick(Sender: TObject);
    procedure BSTARTScaNClick(Sender: TObject);
    procedure BSTOPSCANClick(Sender: TObject);
    procedure FormClose(Sender: TObject; var CloseAction: TCloseAction);
    procedure FormCreate(Sender: TObject);
    procedure FormShow(Sender: TObject);
    function SetupRX: boolean;
    function SetupChart: boolean;
  private

  public

  end;

var
  FSCAN: TFSCAN;
  StopScan: boolean = FALSE;
  SaveSMeterTimer: boolean;

// Declared here for i18n
resourcestring
  CANTCREATEFILE = 'Can''t create file %s.';
  CANTOPENFILE = 'Can''t open file %s.';
  CANTWRITEFILE = 'Can''t write file %s.';
  CANTREADFILE = 'Can''t read file %s.';
  INVALIDSTARTFREQ = 'Invalid start frequency';
  INVALIDSTOPFREQ = 'Invalid stop frequency';
  INVALIDSTEPFREQ = 'Invalid frequency step';
  STARTSCAN = 'Start scan';
  SOMECOMMANDSFAILED = 'One or more commands failed.';
  SCANNING = 'Scanning%s(%d/%d)';
  STOPGREATERSTART = 'Stop frequency must be greater that start frequency.';
  SCANCSVFILENAME = 'Scan-%s-%d-%d.csv';
  SETUPNOTFOUND = 'SETUP line not found, not loading file.';

implementation

{$R *.lfm}

uses EKD500;

{ TFSCAN }

// Setup RX for scan
function TFSCAN.SetupRX: boolean;

var tunestep: integer;
    df,sf: real;
    Ok: boolean = TRUE;
    s: string;

begin
    // Check valid scan data
    if IEFSTART.Value > MAXRXF then begin
      ShowMessage(INVALIDSTARTFREQ);
      exit(FALSE);
    end;
    if IEFSTOP.Value > MAXRXF then begin
      ShowMessage(INVALIDSTOPFREQ);
      exit(FALSE);
    end;
    if IEFSTEP.Value > MAXRXF then begin
      ShowMessage(INVALIDSTEPFREQ);
      exit(FALSE);
    end;

    //Set mode D
    Ok := Ok and EKD.SetRemoteMode(MODE_D, NOFLUSH);

    // Set step value
    tunestep := IEFSTEP.Value div 10; // Make 10 Hz units
    if tunestep = 0 then tunestep := 1; // min 10 Hz
    IEFSTEP.Value := 10*tunestep;  // show rounded value
    df := tunestep/100.0;
    str(df:4:2, s);
    if s[4] = '0' then delete(s,4,1);
    Ok := Ok and EKD.SendCommandD(Key_DF,s+'e');

    // Set mode
    Ok := Ok and EKD.SendCommandD(Key_Mod,IntToStr(CBMODE.ItemIndex+1));

    // Set bandwidth
    Ok := Ok and EKD.SendCommandD(Key_B,IntToStr(CBBANDW.ItemIndex+1));

    // Set AGC
    Ok := Ok and EKD.SendCommandD(Key_GC,IntToStr(CBAGC.ItemIndex));

    // Set start frequency
    sf := IEFSTART.Value / 1000.0;
    str(sf:0:2,s);
    Ok := Ok and EKD.SendCommandD(Key_F, s + 'e');

    // Check if setup succeeded
    if not Ok then
      ShowMessage(SOMECOMMANDSFAILED)
    else begin
      // All ok, so save status timer status and disable it
      SaveSMeterTimer := EKD.TSMeter.Enabled;
      EKD.TSMeter.Enabled := FALSE;
      Enable_Status := TRUE;
    end;

    SetupRX := Ok;
end;

// Check bounds and setup chart X axis for selected start/stop frequencies
function TFSCAN.SetupChart: boolean;

begin
  // Make 10 Hz multiples
  IEFSTART.Value := 10*(IEFSTART.Value div 10);
  IEFSTOP.Value := 10*(IEFSTOP.Value div 10);
  if IEFSTOP.Value <= IEFSTART.Value then begin
    ShowMessage(STOPGREATERSTART);
    Exit(FALSE);
  end else
    SetupChart := TRUE;

  // Setup chart
  with ScanPlot do begin
    AxisList[1].Range.Max := Double(IEFSTOP.Value)/1e6;
    AxisList[1].Range.Min := Double(IEFSTART.Value)/1e6;
    ScanPlot.Bottomaxis.Range.Usemin := TRUE;
    ScanPlot.BottomAxis.Range.Usemax := TRUE;
  end;
end;

// Start scan button
procedure TFSCAN.BSTARTSCANClick(Sender: TObject);

var level, freq, fstep, fstop: double;
     {$IFDEF TESTCHART}
    x, y: double;
    {$ENDIF}
    nsamples: integer;
    Ok: boolean = TRUE;

begin
  if not SetupChart then exit;
  if not SetupRX then exit;
  StopScan := FALSE;
  freq := Double(IEFStart.Value)/1e6;
  fstop := Double(IEFSTOP.Value)/1e6;
  fstep := Double(IEFSTEP.Value)/1e6;
  nsamples := (IEFSTOP.Value - IEFSTART.Value) div IEFSTEP.Value;
  {$IFDEF TESTCHART}
  y:=0;
  randomize;
  repeat
    BSTARTSCAN.LCaption.Caption := Format(SCANNING,
      [LineEnding,SignalLevel.Count,nsamples]);
    x := Random(10000);
    if x >= 9900 then x := Random(100000);
    y := 0.3*(-y+x/100);
    level :=  Double(Trunc(abs(y)));
    if level > 120 then level := 120;
    SignalLevel.AddXY(freq,level);
    freq := freq + fstep;
    Application.ProcessMessages;
  until (freq >= fstop) or StopScan or Error;
  {$ELSE}
  // Dummy level read
  In_Status := TRUE;
  EKD.SetRemoteMode(Mode_C, FLUSH_INPUT);
  sleep(TimeOneCharMs+10);
  EKD.SendCommandC(Key_GC, statusG, DONT_REENABLE);
  EKD.SendCommandC(Key_E, statusC, REENABLE);
  In_Status := FALSE;
  // Do scan
  repeat
    BSTARTSCAN.LCaption.Caption := Format(SCANNING,
      [LineEnding,SignalLevel.Count,nsamples]);

    // read level
    EKD.SetRemoteMode(Mode_C, FLUSH_INPUT);
    sleep(TimeOneCharMs+10);
    statusG := '';
    In_Status := TRUE;
    Ok := EKD.SendCommandC(Key_GC, statusG, DONT_REENABLE);
    Ok := Ok and EKD.SendCommandC(Key_E, statusC, REENABLE);
    In_Status := FALSE;

    {$IFDEF DEBUGLEVEL}
    EKD.Message('statusG = '+ statusG);
    {$ENDIF}
    if statusG<>'' then
      level := Double(2*StrToInt(copy(statusG,1,2)))
    else
      level := 42;
    SignalLevel.AddXY(freq,level);

    // tune to next frequency
    freq := freq + fstep;
    Ok := Ok and EKD.SetRemoteMode(MODE_D, NOFLUSH);
    Ok := Ok and EKD.SendCommandD(Tune_Up,'');
    if IEFSTEP.Value >= 1000 then sleep(100);
    Application.ProcessMessages;
  until (freq >= fstop) or StopScan or not Ok;
  {$ENDIF}
  BSTARTSCAN.LCaption.Caption := STARTSCAN;
end;

// Stop scan button
procedure TFSCAN.BSTOPSCANClick(Sender: TObject);
begin
  StopScan := TRUE;
  BSTARTSCAN.LCaption.Caption := STARTSCAN;
end;

procedure TFSCAN.FormClose(Sender: TObject; var CloseAction: TCloseAction);
begin
    BCLOSEWClick(Self);
    CloseAction := caNone;
end;

// Initializations at startup
procedure TFSCAN.FormCreate(Sender: TObject);
begin
  SaveSMeterTimer := EKD.TSmeter.Enabled;
  EKD.TSmeter.Enabled := FALSE;
end;

procedure TFSCAN.FormShow(Sender: TObject);
begin
  // Set comboboxes to first item
  CBMODE.ItemIndex := 0;
  CBBANDW.ItemIndex := 0;
  CBAGC.ItemIndex := 0;
end;

// Clear scan button
procedure TFSCAN.BCLEARSCANClick(Sender: TObject);
begin
  SignalLevel.Clear;
  ScanPlot.Bottomaxis.Range.Min := 0;
  ScanPlot.BottomAxis.Range.Max := 30;
  ScanPlot.Bottomaxis.Range.Usemin := TRUE;
  ScanPlot.BottomAxis.Range.Usemax := TRUE;
end;

// Save scan data
procedure TFSCAN.BSAVESCANClick(Sender: TObject);

var i: integer;
    f: textfile;
    s: string;
    freq,level: Double;

begin
  DSAVECSV.Filename := Format(SCANCSVFILENAME,
                    [FormatDateTime('DD-MM-YYYY-hh-mm-ss',now),IEFSTART.Value,
                    IEFSTOP.Value]);

  DSAVECSV.Initialdir := BandScansDir;
  if DSAVECSV.Execute then begin
    try
      AssignFile(f,DSAVECSV.Filename);
      Rewrite(f);
    except
      ShowMessage(Format(CANTCREATEFILE, [DSAVECSV.Filename]));
    end;
    // write this scan configuration
    WriteLn(f,Format('"SETUP";"%d:%d:%d:%d:%d:%d:"', [IEFSTART.Value, IEFSTOP.Value,
              IEFSTEP.Value,CBMode.ItemIndex,CBBANDW.ItemIndex,CBAGC.ItemIndex]));

    // Save data points
    for i := 0 to SignalLevel.Count - 1 do begin
      freq := SignalLevel.GetXValue(i);
      level := SignalLevel.GetYValue(i);
      s := FormatFloat('00.000000',freq)+';'+FormatFloat('00',level);
      try
        WriteLn(f,s);
      except
        ShowMessage(Format(CANTWRITEFILE, [DSAVECSV.Filename]));
      end;
    end;
    CloseFile(f);
  end;
end;

// Load & display CSV data saved
procedure TFSCAN.BLOADCSVClick(Sender: TObject);

var f: textfile;
    s,st: string;
    p: integer;
    freq, level: Double;

begin
  DOPENCSV.Initialdir := BandScansDir;
  ScanPlot.Bottomaxis.Range.Usemin := FALSE;
  ScanPlot.BottomAxis.Range.Usemax := FALSE;
  if DOPENCSV.Execute then begin
    SignalLevel.Clear;
    try
      AssignFile(f,DOPENCSV.Filename);
      Reset(f);
    except
      ShowMessage(Format(CANTOPENFILE, [DOPENCSV.Filename]));
    end;

    // Read and set up scan configuration
    try
      ReadLn(f,s);
    except
      ShowMessage(Format(CANTREADFILE, [DOPENCSV.Filename]));
    end;
     // First line is something like
    // "SETUP";"10100000:10150000:100:0:0:0:"
    if Pos('SETUP',s) > 0 then begin
      p := pos(';',s);
      if p > 0 then begin
        s := copy(s, p+2, Length(s)-1);
      end;
      // Parse configuration fields
      st :=  ExtractDelimited(1,s,[':']);
      IEFSTART.Value := StrToInt(st);

      st := ExtractDelimited(2,s,[':']);
      IEFSTOP.Value := StrToInt(st);

      st := ExtractDelimited(3,s,[':']);
      IEFSTEP.Value := StrToInt(st);

      st := ExtractDelimited(4,s,[':']);
      CBMODE.ItemIndex := StrToInt(st);
      case CBMODE.ItemIndex of
        0,1: CBBANDW.Enabled := TRUE;
        2,3: CBBANDW.Enabled := FALSE;
      end;

      st := ExtractDelimited(5,s,[':']);
      CBBANDW.ItemIndex := StrToInt(st);

      st := ExtractDelimited(6,s,[':','"']);
      CBAGC.ItemIndex := StrToInt(st);
    end else begin
      ShowMessage(SETUPNOTFOUND);
      exit;
    end;
    // Read and show scan data
    while not Eof(f) do begin
      try
        ReadLn(f,s);
      except
        ShowMessage(Format(CANTREADFILE, [DOPENCSV.Filename]));
      end;
      p := Pos(';',s);
      if p > 0 then begin
        freq := LazUtilities.StrToDouble(copy(s,1,p-1));
        level := LazUtilities.StrToDouble(Copy(s,p+1,Length(s)));
        SignalLevel.AddXY(freq,level);
      end;
    end;
    CloseFile(f);
  end;
end;

// Close window button
procedure TFSCAN.BCLOSEWClick(Sender: TObject);

begin
   StopScan := TRUE;
   FSCAN.Hide;
   with EKD do begin
      RestoreState;
      TSMeter.Enabled := SaveSMeterTimer;
      Enable_Status := SaveSMeterTimer
   end;
end;

end.

