{
    This file is part of RA3701Control

    RACAL RA3701 Control program

    Copyright (C) 2025,2026 G. Perotti, I1EPJ, i1epj@aricasale.it

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/.

                                    * * *

    Communication parameters select dialog
}

unit userparms;

{$mode objfpc}{$H+}

{$INCLUDE defines.inc}

interface

uses
  Classes, SysUtils, FileUtil, Forms, Controls, Graphics, Dialogs, StdCtrls,
  Grids, Math;

const
  PROTO_NONE = 0;
  PROTO_CTR  = 1;
  PROTO_RTS  = 2;

  PARITY_NONE = 0;
  PARITY_EVEN = 1;
  PARITY_ODD  = 2;

type

  { TFSerParms }

  TFSerParms = class(TForm)
    BOK: TButton;
    BCANCEL: TButton;
    CBCRC: TComboBox;
    CBNADCH: TComboBox;
    CBNRX: TComboBox;
    CBPORT: TComboBox;
    CBPROTOCOL: TComboBox;
    CBLCC: TComboBox;
    CBSPEED: TComboBox;
    CBPARITY: TComboBox;
    CBWAITCTS: TCheckBox;
    LBCRC: TLabel;
    LBNADDRC: TLabel;
    LBNRX: TLabel;
    LBRXCONF: TLabel;
    LBPORT: TLabel;
    LBPROTOCOL: TLabel;
    LBLCC: TLabel;
    LBSPEED: TLabel;
    LBPARITY: TLabel;
    SGRXADDRESSES: TStringGrid;
    procedure BCANCELClick(Sender: TObject);
    procedure BOKClick(Sender: TObject);
    procedure CBNADCHChange(Sender: TObject);
    procedure CBNRXChange(Sender: TObject);
    procedure FormShow(Sender: TObject);
  private
    { private declarations }
  public
    { public declarations }
  end;

  // The communication parameters in a record
  TCommParms = record
    SerPortName: string;
    SerPortSpeed: integer;
    SerPortParity: integer;
    SerPortProtocol: integer;
    WaitCTSinRTSProtocol: boolean;
    NumLCCChars: integer;
    NumAddrChars: integer;
    CRCEnabled: boolean;
    NumRX: integer;
    RXAddresses: array[1..99] of integer;
  end;

var
  FSerParms: TFSerParms;
  MaxNumRX: integer;
  CommParms: TCommParms;

// For i18n
resourcestring
  YES = 'YES';
   NO = 'NO';
 NONE = 'NONE';
  CTR = 'CTR';
  RTS = 'RTS';

implementation

{$R *.lfm}

{ TFSerParms }

uses ura3701;

// OK Button handling.
procedure TFSerParms.BOKClick(Sender: TObject);

var i,addr,MaxRXaddr: integer;
  err,Ok: boolean;

begin
  // Reads and checks parameters and stores them in the CommParms record.
  with CommParms do begin
    {$IFDEF LINUX}
    if pos('/dev',CBPORT.Text) = 0 then
      SerPortName := '/dev/'+CBPORT.Text
    else
      SerPortName := CBPORT.Text;
    {$ENDIF}
    {$IFDEF WINDOWS}
    // The \\.\ syntax is optional for ports 1..9 but is needed
    // for ports > 9, so insert it anyway if not present
    if pos('\\.\', CBPORT.Text) = 0 then
      SerPortName := '\\.\' + CBPORT.Text
    else
      SerPortName := CBPORT.Text;
    {$ENDIF}
    SerPortSpeed := StrToInt(CBSPEED.Text);
    SerPortParity := CBPARITY.ItemIndex;
    SerPortProtocol := CBPROTOCOL.ItemIndex;
    NumLCCChars := CBLCC.ItemIndex;
    NumAddrChars := CBNADCH.ItemIndex;
    MaxRXAddr := Trunc(Power(10,NumAddrChars));
    CRCEnabled :=  (CBCRC.ItemIndex > 0);
    if NumAddrChars > 0 then
      NumRX := StrToInt(CBNRX.Text)
    else
      NumRX := 1;
    RA3701.LBRXNUM.Caption := 'RX#';
    RA3701.SPRXNUM.MinValue := 1;
    RA3701.SPRXNUM.MaxValue := NumRX;

    // Check for valid RX adddresses
    OK:= TRUE;
    if NumAddrChars > 0 then begin
      for i := 1 to NumRX do begin
        err := FALSE;
        if TryStrToInt(SGRXADDRESSES.Cells[1,i],addr) then begin
          if addr < MaxRXAddr then begin
            RXaddresses[i] := addr
          end else err := TRUE;
        end else err := TRUE;
        if err then begin
          SGRXADDRESSES.Cells[1,i] := SGRXADDRESSES.Cells[1,i] + '??';
          Ok := FALSE;
        end;
      end;
      if not Ok then exit;
    end;
  end;

  // Read waiting for CTS in RTS protocol option
  CommParms.WaitCTSinRTSProtocol := CBWAITCTS.Checked;

  RA3701.SaveConfig;
  FSerParms.Hide;
end;

// Cancel button handling.
// Hide window without doing anything
procedure TFSerParms.BCANCELClick(Sender: TObject);
begin
  FSerParms.Hide;
end;

// Handles change in number of address characters
procedure TFSerParms.CBNADCHChange(Sender: TObject);

var i,oldindex: integer;

begin
  case CBNADCH.ItemIndex of
    0: begin
      SGRXADDRESSES.Enabled := FALSE;
      SGRXADDRESSES.RowCount := 2;
      CBNRX.Enabled := FALSE;
      LBNRX.Enabled := FALSE;
      LBRXCONF.Enabled := FALSE;
      MaxNumRX := 1;
      CommParms.NumRX := 1;
    end;
    1: begin
      SGRXADDRESSES.Enabled := TRUE;
      CBNRX.Enabled := TRUE;
      LBNRX.Enabled := TRUE;
      LBRXCONF.Enabled := TRUE;
      CBNRX.ItemIndex := 1;
      MaxNumRX := 9;
      CommParms.NumRX := 1;
      CBNRX.Text := '1';
      SGRXADDRESSES.RowCount := CBNRX.ItemIndex+2; //StrToInt(CBNRX.Text)+1;
      for i := 1 to CBNRX.ItemIndex+1 do SGRXADDRESSES.Cells[0,i] := IntToStr(i);
    end;
    2: begin
      SGRXADDRESSES.Enabled := TRUE;
      CBNRX.Enabled := TRUE;
      LBNRX.Enabled := TRUE;
      MaxNumRX := 99;
      CommParms.NumRX := 1;
      CBNRX.Text := '1';
      SGRXADDRESSES.RowCount := CBNRX.ItemIndex+2;
      for i := 1 to CBNRX.ItemIndex+1 do SGRXADDRESSES.Cells[0,i] := IntToStr(i);
    end;
    otherwise begin
      SGRXADDRESSES.Enabled := FALSE;
      MaxNumRX := 1;
    end;
  end;
  OldIndex := CBNRX.ItemIndex;
  CBNRX.Clear;
  for i := 1 to MaxNumRX do CBNRX.Items.Add(IntToStr(i));
  CBNRX.ItemIndex := OldIndex;
end;

// Handles changes in number of receivers
procedure TFSerParms.CBNRXChange(Sender: TObject);

var i: integer;
    nRX: integer;

begin
  nRX := StrToInt(CBNRX.Text);
  SGRXADDRESSES.RowCount := nRX+1;
  RA3701.SPRXNUM.MaxValue := nRX;
  for i := 1 to nRX do SGRXADDRESSES.Cells[0,i] := IntToStr(i);
end;

// Things to do when form is shown.
// Reads CommsParms record and set controls default values.
procedure TFSerParms.FormShow(Sender: TObject);

var i: integer = 0;

begin
  RA3701.ReadConfig;
  SGRXADDRESSES.Cells[0,0] := 'RX#';
  SGRXADDRESSES.Cells[1,0] := 'Addr';
  CBLCC.Clear;
  CBLCC.AddItem(NO, nil);
  CBLCC.AddItem(YES, nil);
  CBCRC.Clear;
  CBCRC.AddItem(NO, nil);
  CBCRC.AddItem(YES, nil);
  CBPROTOCOL.Clear;
  CBPROTOCOL.AddItem(NONE,nil);
  CBPROTOCOL.AddItem(CTR, nil);
  CBPROTOCOL.AddItem(RTS, nil);

  {$IFDEF WINDOWS}
  // Set serial port names for Windows
  CBPORT.Clear;
  CBPORT.AddItem('\\.\COM1',nil);
  CBPORT.AddItem('\\.\COM2',nil);
  CBPORT.AddItem('\\.\COM3',nil);
  CBPORT.AddItem('\\.\COM4',nil);
  {$ENDIF}

  // Set controls value from CommParms record
  with CommParms do begin
    CBPORT.Text := SerPortName;
    CBSPEED.Text := IntToStr(SerPortSpeed);
    CBPARITY.ItemIndex := SerPortParity;
    CBPROTOCOL.ItemIndex := SerPortProtocol;
    CBWAITCTS.Checked := WaitCTSinRTSProtocol;
    CBLCC.ItemIndex := NumLCCChars;
    CBNADCH.ItemIndex := NumAddrChars;
    CBNRX.Text := IntToStr(NumRX);
    if CRCEnabled then
      CBCRC.ItemIndex := 1
    else
      CBCRC.ItemIndex := 0;

    case NumAddrChars of
      0: begin
        LBNRX.Enabled := FALSE;
        LBRXCONF.Enabled := FALSE;
        CBNRX.Enabled := FALSE;
        SGRXADDRESSES.Enabled := FALSE;
      end;
      1,2: begin
        LBNRX.Show;
        CBNRX.Enabled := TRUE;
        SGRXADDRESSES.Enabled := TRUE;
        SGRXADDRESSES.RowCount := StrToInt(CBNRX.Text)+1;
        for i := 1 to CommParms.NumRX do begin
          SGRXADDRESSES.Cells[0,i] := IntToStr(i);
          SGRXADDRESSES.Cells[1,i] := IntToSTr(RXAddresses[i]);
        end;
        CBNRX.Items.Clear;
        for i := 1 to Trunc(Power(10,NumAddrChars)) do
          CBNRX.Items.Add(IntToStr(i));
//        CBNRX.ItemIndex := 0;
//        CBNRX.Text := '1';
     end;
    end;
  end;

  {$IFNDEF ENABLE_ALL_COMMPARMS}
  // Disable configuration controls not yet handled
  // Currently only port, speed, parity, address and CRC are honored
  LBLCC.Enabled := FALSE;
  CBLCC.Enabled := FALSE;
  {$ENDIF}
end;

end.

